/*
 * Decompiled with CFR 0.152.
 */
package org.cytoscape.search.internal.index;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexableField;
import org.apache.lucene.index.MultiBits;
import org.apache.lucene.queryparser.classic.QueryParser;
import org.apache.lucene.util.Bits;
import org.cytoscape.event.DebounceTimer;
import org.cytoscape.model.CyColumn;
import org.cytoscape.model.CyIdentifiable;
import org.cytoscape.model.CyNetwork;
import org.cytoscape.model.CyNetworkTableManager;
import org.cytoscape.model.CyTable;
import org.cytoscape.model.events.ColumnDeletedEvent;
import org.cytoscape.model.events.ColumnDeletedListener;
import org.cytoscape.model.events.ColumnNameChangedEvent;
import org.cytoscape.model.events.ColumnNameChangedListener;
import org.cytoscape.model.events.RowSetRecord;
import org.cytoscape.model.events.RowsCreatedEvent;
import org.cytoscape.model.events.RowsCreatedListener;
import org.cytoscape.model.events.RowsDeletedEvent;
import org.cytoscape.model.events.RowsDeletedListener;
import org.cytoscape.model.events.RowsSetEvent;
import org.cytoscape.model.events.RowsSetListener;
import org.cytoscape.model.events.TableAboutToBeDeletedEvent;
import org.cytoscape.model.events.TableAboutToBeDeletedListener;
import org.cytoscape.model.events.TableAddedEvent;
import org.cytoscape.model.events.TableAddedListener;
import org.cytoscape.model.subnetwork.CyRootNetwork;
import org.cytoscape.search.internal.index.Index;
import org.cytoscape.search.internal.index.TableIndexer;
import org.cytoscape.search.internal.index.TableType;
import org.cytoscape.search.internal.progress.CompositeProgressMonitor;
import org.cytoscape.search.internal.progress.ProgressMonitor;
import org.cytoscape.search.internal.progress.ProgressViewer;
import org.cytoscape.service.util.CyServiceRegistrar;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SearchManager
implements TableAddedListener,
TableAboutToBeDeletedListener,
ColumnDeletedListener,
ColumnNameChangedListener,
RowsSetListener,
RowsCreatedListener,
RowsDeletedListener {
    private static final Logger logger = LoggerFactory.getLogger((String)"org.cytoscape.application.userlog");
    public static final String INDEX_FIELD = "CY_SEARCH2_INDEX";
    public static final String TYPE_FIELD = "CY_SEARCH2_TYPE";
    private final CyServiceRegistrar registrar;
    private final Path baseDir;
    private final ExecutorService executorService;
    private final List<ProgressViewer> progressViewers = new ArrayList<ProgressViewer>();
    private final Map<Long, Index> tableIndexMap = new ConcurrentHashMap<Long, Index>();
    private final DebounceTimer columnChangeDebounceTimer = new DebounceTimer();

    public SearchManager(CyServiceRegistrar registrar, Path baseDir) {
        this.registrar = registrar;
        this.baseDir = Objects.requireNonNull(baseDir);
        this.executorService = Executors.newSingleThreadExecutor(r -> {
            Thread thread = Executors.defaultThreadFactory().newThread(r);
            thread.setName("search2-" + thread.getName());
            return thread;
        });
    }

    public boolean isReady(Long suid) {
        return this.tableIndexMap.containsKey(suid);
    }

    public boolean isReady(CyTable tab) {
        return this.isReady(tab.getSUID());
    }

    public boolean isReady(CyTable tab1, CyTable tab2) {
        return this.isReady(tab1.getSUID()) && this.isReady(tab2.getSUID());
    }

    public void addProgressViewer(ProgressViewer viewer) {
        this.progressViewers.add(viewer);
    }

    private Path getIndexPath(CyIdentifiable ele) {
        return this.baseDir.resolve("index_" + ele.getSUID());
    }

    public QueryParser getQueryParser(CyTable table) {
        Index index = this.tableIndexMap.get(table.getSUID());
        return index == null ? null : index.getQueryParser(table);
    }

    public IndexReader getIndexReader(CyTable table) throws IOException {
        return this.tableIndexMap.get(table.getSUID()).getIndexReader();
    }

    private ProgressMonitor getProgressMonitor(CyTable table, String suffix) {
        String name;
        if (this.progressViewers.isEmpty()) {
            return ProgressMonitor.nullMonitor();
        }
        String string = name = suffix == null ? table.getTitle() : table.getTitle() + " (" + suffix + ")";
        if (this.progressViewers.size() == 1) {
            return this.progressViewers.get(0).addProgress(name);
        }
        List<ProgressMonitor> monitors = this.progressViewers.stream().map(pv -> pv.addProgress(name)).collect(Collectors.toList());
        return new CompositeProgressMonitor(monitors);
    }

    private TableType getTableType(CyTable table) {
        CyNetworkTableManager networkTableManager = (CyNetworkTableManager)this.registrar.getService(CyNetworkTableManager.class);
        CyNetwork network = networkTableManager.getNetworkForTable(table);
        if (network == null) {
            return null;
        }
        if (network instanceof CyRootNetwork) {
            return null;
        }
        if (network.getDefaultNodeTable().equals(table)) {
            return TableType.NODE;
        }
        if (network.getDefaultEdgeTable().equals(table)) {
            return TableType.EDGE;
        }
        return null;
    }

    public boolean isIndexable(CyTable table) {
        return this.getTableType(table) != null;
    }

    public void handleEvent(TableAddedEvent e) {
        CyTable table = e.getTable();
        TableType type = this.getTableType(table);
        if (type == null) {
            return;
        }
        this.addTable(table, type);
    }

    public Future<?> addTable(CyTable table, TableType type) {
        ProgressMonitor pm = this.getProgressMonitor(table, null);
        Path path = this.getIndexPath((CyIdentifiable)table);
        Index index = new Index(table.getSUID(), type, path);
        return this.executorService.submit(() -> {
            Long suid = table.getSUID();
            if (this.tableIndexMap.containsKey(suid)) {
                return;
            }
            try {
                TableIndexer.indexTable(index, table, pm);
            }
            catch (IOException e) {
                logger.error("Error indexing table: " + suid, (Throwable)e);
            }
            finally {
                pm.done();
            }
            this.tableIndexMap.put(suid, index);
        });
    }

    public void handleEvent(TableAboutToBeDeletedEvent e) {
        CyTable table = e.getTable();
        this.removeTable(table);
    }

    public Future<?> removeTable(CyTable table) {
        Long suid = table.getSUID();
        return this.executorService.submit(() -> {
            Index index = this.tableIndexMap.remove(suid);
            try {
                SearchManager.deleteIndex(index);
            }
            catch (IOException e) {
                logger.error("Error deleting table index: " + suid, (Throwable)e);
            }
        });
    }

    private static void deleteIndex(Index index) throws IOException {
        if (index == null) {
            return;
        }
        IndexWriter writer = index.getWriter();
        writer.deleteAll();
        writer.commit();
        writer.close();
        SearchManager.deleteFolder(index.getPath().toFile());
    }

    private static void deleteFolder(File folder) {
        File[] files = folder.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    SearchManager.deleteFolder(file);
                    continue;
                }
                file.delete();
            }
        }
        folder.delete();
    }

    public void disposeAll() throws Exception {
        Future<?> future = this.executorService.submit(() -> {
            for (Index index : this.tableIndexMap.values()) {
                try {
                    SearchManager.deleteIndex(index);
                }
                catch (IOException e) {
                    logger.error("Error deleting table index", (Throwable)e);
                }
            }
        });
        future.get(5L, TimeUnit.MINUTES);
        this.tableIndexMap.clear();
        this.executorService.shutdownNow();
    }

    public void handleEvent(RowsSetEvent e) {
        Set cols = e.getColumns();
        if (cols.size() == 1 && cols.contains("selected")) {
            return;
        }
        CyTable table = (CyTable)e.getSource();
        if (!this.isIndexable(table)) {
            return;
        }
        CyColumn keyCol = table.getPrimaryKey();
        String keyName = keyCol.getName();
        Class keyType = keyCol.getType();
        HashSet<Object> keys = new HashSet<Object>();
        for (RowSetRecord rowSetRecord : e.getPayloadCollection()) {
            String column = rowSetRecord.getColumn();
            if ("selected".equals(column)) continue;
            Object key = rowSetRecord.getRow().get(keyName, keyType);
            keys.add(key);
        }
        this.updateRows(table, keys);
    }

    public void handleEvent(RowsDeletedEvent e) {
        CyTable table = (CyTable)e.getSource();
        if (!this.isIndexable(table)) {
            return;
        }
        Collection keys = e.getKeys();
        this.updateRows(table, keys);
    }

    public void handleEvent(RowsCreatedEvent e) {
        CyTable table = (CyTable)e.getSource();
        if (!this.isIndexable(table)) {
            return;
        }
        Collection keys = e.getPayloadCollection();
        this.updateRows(table, keys);
    }

    public Future<?> updateRows(CyTable table, Collection<? extends Object> keys) {
        Long suid = table.getSUID();
        ProgressMonitor pm = this.getProgressMonitor(table, "update rows");
        return this.executorService.submit(() -> {
            try {
                Index index = this.tableIndexMap.get(suid);
                if (index != null) {
                    TableIndexer.updateRows(index, table, keys, pm);
                }
            }
            catch (IOException e) {
                logger.error("Error indexing table: " + suid, (Throwable)e);
            }
            finally {
                pm.done();
            }
        });
    }

    public void handleEvent(ColumnDeletedEvent e) {
        CyTable table = (CyTable)e.getSource();
        if (this.isIndexable(table)) {
            this.columnChangeDebounceTimer.debounce(() -> this.reindexTable(table));
        }
    }

    public void handleEvent(ColumnNameChangedEvent e) {
        CyTable table = (CyTable)e.getSource();
        if (this.isIndexable(table)) {
            this.columnChangeDebounceTimer.debounce(() -> this.reindexTable(table));
        }
    }

    public Future<?> reindexTable(CyTable table) {
        Long suid = table.getSUID();
        ProgressMonitor pm = this.getProgressMonitor(table, "reindex table");
        return this.executorService.submit(() -> {
            try {
                Index index = this.tableIndexMap.get(suid);
                if (index != null) {
                    ProgressMonitor[] subPms = pm.split(1.0, 10.0);
                    IndexWriter writer = index.getWriter();
                    writer.deleteAll();
                    subPms[0].done();
                    TableIndexer.indexTable(index, table, subPms[1]);
                    writer.commit();
                    subPms[1].done();
                }
            }
            catch (IOException e) {
                logger.error("Error indexing table: " + suid, (Throwable)e);
            }
            finally {
                pm.done();
            }
        });
    }

    protected int getDocumentCount(CyTable table) throws IOException {
        Index index = this.tableIndexMap.get(table.getSUID());
        try (IndexReader indexReader = index.getIndexReader();){
            int n = indexReader.numDocs();
            return n;
        }
    }

    protected void printIndex(CyTable table) throws IOException {
        Index index = this.tableIndexMap.get(table.getSUID());
        try (IndexReader indexReader = index.getIndexReader();){
            System.out.println("All Documents in Lucene Index");
            Bits liveDocs = MultiBits.getLiveDocs((IndexReader)indexReader);
            for (int i = 0; i < indexReader.maxDoc(); ++i) {
                if (liveDocs != null && !liveDocs.get(i)) continue;
                Document doc = indexReader.document(i);
                System.out.print(doc.get(INDEX_FIELD) + " - ");
                boolean first = true;
                for (IndexableField field : doc.getFields()) {
                    if (!first) {
                        System.out.print(", ");
                    }
                    System.out.print(field.name() + ":" + doc.get(field.name()));
                    first = false;
                }
                System.out.println();
            }
            System.out.println();
        }
    }
}

